<?php
/**
 * Alternative ping implementation using PHP sockets
 * Solusi untuk masalah ping utilities tidak tersedia di container
 */

require_once '../config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

/**
 * PHP-based ping using socket connection
 * Alternative ketika ping command tidak tersedia
 */
function socketPing($host, $port = null, $timeout = 3) {
    $start_time = microtime(true);
    
    // Clean host from protocol
    $host = preg_replace('/^https?:\/\//', '', $host);
    
    // If no port specified, try different ports based on common services
    $test_ports = $port ? [$port] : [80, 443, 22, 23];
    
    $result = [
        'host' => $host,
        'status' => 'down',
        'response_time' => null,
        'packet_loss' => 100,
        'timestamp' => date('Y-m-d H:i:s'),
        'method' => 'socket',
        'tested_ports' => [],
        'successful_port' => null
    ];
    
    foreach ($test_ports as $test_port) {
        $port_start = microtime(true);
        
        // Try to connect via socket
        $socket = @fsockopen($host, $test_port, $errno, $errstr, $timeout);
        
        if ($socket) {
            fclose($socket);
            $port_time = round((microtime(true) - $port_start) * 1000, 2);
            
            $result['status'] = 'up';
            $result['response_time'] = $port_time;
            $result['packet_loss'] = 0;
            $result['successful_port'] = $test_port;
            $result['tested_ports'][] = ['port' => $test_port, 'status' => 'open', 'time' => $port_time];
            break; // Stop at first successful connection
        } else {
            $port_time = round((microtime(true) - $port_start) * 1000, 2);
            $result['tested_ports'][] = ['port' => $test_port, 'status' => 'closed', 'time' => $port_time, 'error' => $errstr];
        }
    }
    
    $total_time = round((microtime(true) - $start_time) * 1000, 2);
    $result['total_time'] = $total_time;
    
    return $result;
}

/**
 * HTTP-based connectivity test
 * Alternative ping using HTTP HEAD request
 */
function httpPing($host, $timeout = 5) {
    $start_time = microtime(true);
    
    // Clean and prepare URL
    $host = preg_replace('/^https?:\/\//', '', $host);
    $urls = [
        "http://$host",
        "https://$host"
    ];
    
    $result = [
        'host' => $host,
        'status' => 'down',
        'response_time' => null,
        'packet_loss' => 100,
        'timestamp' => date('Y-m-d H:i:s'),
        'method' => 'http',
        'http_code' => null,
        'tested_urls' => []
    ];
    
    foreach ($urls as $url) {
        $url_start = microtime(true);
        
        $context = stream_context_create([
            'http' => [
                'method' => 'HEAD',
                'timeout' => $timeout,
                'ignore_errors' => true,
                'follow_location' => false
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        $url_time = round((microtime(true) - $url_start) * 1000, 2);
        
        // Get HTTP response code
        $http_code = null;
        if (isset($http_response_header[0])) {
            preg_match('/HTTP\/\d\.\d\s+(\d+)/', $http_response_header[0], $matches);
            $http_code = isset($matches[1]) ? intval($matches[1]) : null;
        }
        
        $result['tested_urls'][] = [
            'url' => $url,
            'http_code' => $http_code,
            'time' => $url_time,
            'success' => ($http_code !== null && $http_code < 500)
        ];
        
        // Consider it successful if we get any HTTP response (even error codes)
        if ($http_code !== null) {
            $result['status'] = 'up';
            $result['response_time'] = $url_time;
            $result['packet_loss'] = 0;
            $result['http_code'] = $http_code;
            break;
        }
    }
    
    $total_time = round((microtime(true) - $start_time) * 1000, 2);
    $result['total_time'] = $total_time;
    
    return $result;
}

/**
 * Combined ping test using multiple methods
 */
function smartPing($host, $timeout = 3) {
    $results = [];
    
    // Method 1: Socket ping (fastest, checks connectivity)
    $socket_result = socketPing($host, null, $timeout);
    $results['socket'] = $socket_result;
    
    // Method 2: HTTP ping (more comprehensive, checks web services)
    $http_result = httpPing($host, $timeout);
    $results['http'] = $http_result;
    
    // Determine final status
    $final_status = 'down';
    $best_response_time = null;
    $method_used = 'none';
    
    if ($socket_result['status'] === 'up') {
        $final_status = 'up';
        $best_response_time = $socket_result['response_time'];
        $method_used = 'socket';
    } elseif ($http_result['status'] === 'up') {
        $final_status = 'up';
        $best_response_time = $http_result['response_time'];
        $method_used = 'http';
    }
    
    return [
        'host' => $host,
        'status' => $final_status,
        'response_time' => $best_response_time,
        'packet_loss' => $final_status === 'up' ? 0 : 100,
        'timestamp' => date('Y-m-d H:i:s'),
        'method_used' => $method_used,
        'all_results' => $results
    ];
}

// Handle API requests
try {
    $action = $_GET['action'] ?? $_POST['action'] ?? 'ping_single';
    
    switch ($action) {
        case 'ping_single':
            $host = $_GET['host'] ?? $_POST['host'] ?? '';
            $method = $_GET['method'] ?? 'smart'; // socket, http, smart
            
            if (empty($host)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Host parameter is required'
                ]);
                break;
            }
            
            switch ($method) {
                case 'socket':
                    $result = socketPing($host);
                    break;
                case 'http':
                    $result = httpPing($host);
                    break;
                case 'smart':
                default:
                    $result = smartPing($host);
                    break;
            }
            
            echo json_encode([
                'success' => true,
                'ping_result' => $result
            ]);
            break;
            
        case 'test_methods':
            $host = $_GET['host'] ?? '8.8.8.8';
            
            $results = [
                'socket' => socketPing($host),
                'http' => httpPing($host),
                'smart' => smartPing($host)
            ];
            
            echo json_encode([
                'success' => true,
                'host' => $host,
                'results' => $results,
                'comparison' => [
                    'fastest_method' => 'socket',
                    'most_reliable' => 'smart',
                    'recommendation' => 'Use smart method for best results'
                ]
            ]);
            break;
            
        default:
            echo json_encode([
                'success' => false,
                'message' => 'Invalid action. Use: ping_single, test_methods'
            ]);
            break;
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
}
?>
