<?php
/**
 * Auto Monitor Script for FTTH NMS
 * Script untuk monitoring otomatis status perangkat dengan ping
 */

require_once 'config/database.php';

// Set time limit untuk script monitoring
set_time_limit(120); // 2 menit maximum

// Function untuk output dengan timestamp
function logOutput($message) {
    echo "[" . date('Y-m-d H:i:s') . "] " . $message . "\n";
}

// Import ping functions
function trySystemPing($host) {
    $os = strtoupper(substr(PHP_OS, 0, 3));
    
    if ($os === 'WIN') {
        $command = "ping -n 1 -w 3000 " . escapeshellarg($host);
    } else {
        $command = "ping -c 1 -W 3 " . escapeshellarg($host);
    }
    
    $output = array();
    $return_code = 0;
    
    exec($command, $output, $return_code);
    $output_text = implode("\n", $output);
    
    if ($return_code === 127 || strpos($output_text, 'command not found') !== false) {
        return array('success' => false, 'error' => 'Ping command not available', 'response_time_ms' => null);
    }
    
    if ($return_code === 0) {
        $response_time_ms = null;
        
        if ($os === 'WIN') {
            if (preg_match('/time<1ms/i', $output_text)) {
                $response_time_ms = 0.5;
            } elseif (preg_match('/time[=](\d+(?:\.\d+)?)ms/i', $output_text, $matches)) {
                $response_time_ms = floatval($matches[1]);
            }
        } else {
            if (preg_match('/time=(\d+(?:\.\d+)?)\s*ms/i', $output_text, $matches)) {
                $response_time_ms = floatval($matches[1]);
            }
        }
        
        return array('success' => true, 'response_time_ms' => $response_time_ms ?? 1.0);
    } else {
        return array('success' => false, 'error' => $output_text, 'response_time_ms' => null);
    }
}

function trySocketPing($host, $timeout = 3) {
    $test_ports = [80, 443, 22, 23];
    
    foreach ($test_ports as $port) {
        $start_time = microtime(true);
        $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
        
        if ($socket) {
            fclose($socket);
            $response_time = round((microtime(true) - $start_time) * 1000, 1);
            return array('success' => true, 'response_time_ms' => $response_time, 'method' => 'socket', 'port' => $port);
        }
    }
    
    return array('success' => false, 'error' => "Cannot connect to $host via socket", 'response_time_ms' => null);
}

function pingHost($host, $timeout = 3) {
    $host = preg_replace('/^https?:\/\//', '', $host);
    
    $ping_result = trySystemPing($host);
    
    if (!$ping_result['success']) {
        $socket_result = trySocketPing($host, $timeout);
        
        if ($socket_result['success']) {
            return [
                'success' => true,
                'status' => 'online',
                'response_time' => $socket_result['response_time_ms'],
                'method' => $socket_result['method'] ?? 'socket'
            ];
        } else {
            return [
                'success' => false,
                'status' => 'offline',
                'response_time' => null,
                'error' => $socket_result['error']
            ];
        }
    }
    
    return [
        'success' => true,
        'status' => 'online',
        'response_time' => $ping_result['response_time_ms'],
        'method' => 'system_ping'
    ];
}

try {
    logOutput("🚀 Starting FTTH NMS Auto Monitor");
    
    $database = new Database();
    $db = $database->getConnection();
    
    if (!$db) {
        logOutput("❌ Database connection failed");
        exit(1);
    }
    
    logOutput("✅ Database connection successful");
    
    // Get all items with IP addresses that need monitoring
    $query = "SELECT id, name, ip_address, item_type_id, monitoring_status 
              FROM ftth_items 
              WHERE ip_address IS NOT NULL 
              AND ip_address != '' 
              ORDER BY item_type_id, name";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    logOutput("📋 Found " . count($items) . " items to monitor");
    
    $online_count = 0;
    $offline_count = 0;
    $error_count = 0;
    
    foreach ($items as $item) {
        $item_id = $item['id'];
        $item_name = $item['name'];
        $ip_address = $item['ip_address'];
        
        logOutput("🔍 Testing {$item_name} ({$ip_address})");
        
        // Ping the device
        $ping_result = pingHost($ip_address);
        
        if ($ping_result['success']) {
            $new_status = 'online';
            $response_time = $ping_result['response_time'];
            $method = $ping_result['method'] ?? 'unknown';
            
            logOutput("✅ {$item_name}: ONLINE ({$response_time}ms via {$method})");
            $online_count++;
        } else {
            $new_status = 'offline';
            $response_time = null;
            $error_message = $ping_result['error'] ?? 'Connection failed';
            
            logOutput("❌ {$item_name}: OFFLINE ({$error_message})");
            $offline_count++;
        }
        
        // Update monitoring status in database
        try {
            $update_query = "UPDATE ftth_items 
                           SET monitoring_status = :status,
                               last_ping_time = NOW(),
                               response_time_ms = :response_time,
                               updated_at = NOW()
                           WHERE id = :id";
            
            $update_stmt = $db->prepare($update_query);
            $update_stmt->bindParam(':status', $new_status);
            $update_stmt->bindParam(':response_time', $response_time);
            $update_stmt->bindParam(':id', $item_id);
            
            if ($update_stmt->execute()) {
                logOutput("💾 Status updated for {$item_name}");
            } else {
                logOutput("⚠️ Failed to update status for {$item_name}");
                $error_count++;
            }
            
            // Insert monitoring log entry
            $log_query = "INSERT INTO monitoring_logs (item_id, ping_time, status, response_time_ms, error_message)
                         VALUES (:item_id, NOW(), :status, :response_time, :error_message)";
            
            $log_stmt = $db->prepare($log_query);
            $error_message = $ping_result['error'] ?? null;
            $log_stmt->bindParam(':item_id', $item_id);
            $log_stmt->bindParam(':status', $new_status);
            $log_stmt->bindParam(':response_time', $response_time);
            $log_stmt->bindParam(':error_message', $error_message);
            $log_stmt->execute();
            
        } catch (Exception $e) {
            logOutput("💥 Database error for {$item_name}: " . $e->getMessage());
            $error_count++;
        }
        
        // Small delay to prevent overwhelming the network
        usleep(100000); // 0.1 seconds
    }
    
    logOutput("📊 Monitoring Summary:");
    logOutput("   🟢 Online: {$online_count} devices");
    logOutput("   🔴 Offline: {$offline_count} devices");
    logOutput("   ⚠️ Errors: {$error_count} updates failed");
    logOutput("🏁 Auto monitoring completed successfully");
    
} catch (Exception $e) {
    logOutput("💥 FATAL ERROR: " . $e->getMessage());
    exit(1);
}
?>
