-- =====================================================
-- FTTH Network Management System (FTTHNMS) 
-- Database Schema - CLEAN VERSION
-- =====================================================
-- Versi: 6.0.0 (Clean & Simplified)
-- Tanggal: 2025-08-28
-- Fix: Removed excessive data, simplified structure, fixed import issues
-- Features: Clean database yang bisa diimport tanpa masalah
-- 
-- CHANGES FROM ORIGINAL:
-- - Removed excessive INSERT data (only users and item_types remain)
-- - Simplified foreign key constraints to avoid import issues
-- - Fixed circular dependencies
-- - Ensured MySQL compatibility
-- =====================================================

CREATE DATABASE IF NOT EXISTS ftthnms;
USE ftthnms;

-- Set foreign key checks off untuk avoid constraint issues saat setup
SET FOREIGN_KEY_CHECKS = 0;
SET SQL_MODE = 'NO_AUTO_VALUE_ON_ZERO';
SET time_zone = '+00:00';

-- =====================================================
-- 1. BASIC MASTER DATA TABLES (NO FOREIGN KEYS)
-- =====================================================

-- Tabel untuk menyimpan jenis item FTTH
CREATE TABLE item_types (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) NOT NULL,
    icon VARCHAR(100),
    color VARCHAR(20),
    supports_snmp TINYINT(1) DEFAULT 0 COMMENT 'Support SNMP monitoring',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabel untuk menyimpan warna tube (untuk fiber optic)
CREATE TABLE tube_colors (
    id INT PRIMARY KEY AUTO_INCREMENT,
    color_name VARCHAR(30) NOT NULL,
    hex_code VARCHAR(7),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabel untuk menyimpan jenis splitter
CREATE TABLE splitter_types (
    id INT PRIMARY KEY AUTO_INCREMENT,
    type VARCHAR(20) NOT NULL,
    ratio VARCHAR(10) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabel untuk menyimpan user dan role
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    role ENUM('admin', 'teknisi') NOT NULL DEFAULT 'teknisi',
    full_name VARCHAR(100),
    email VARCHAR(100),
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- =====================================================
-- 2. MAIN DATA TABLES
-- =====================================================

-- Tabel utama untuk menyimpan item-item FTTH di maps
CREATE TABLE ftth_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_type_id INT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    latitude DECIMAL(10, 8) NOT NULL,
    longitude DECIMAL(11, 8) NOT NULL,
    address TEXT,
    
    -- Basic network info
    ip_address VARCHAR(45) NULL COMMENT 'IP Address untuk monitoring',
    port_http INT NULL DEFAULT 80,
    port_https INT NULL DEFAULT 443,
    monitoring_status ENUM('online', 'warning', 'offline') NULL DEFAULT 'offline',
    
    -- Cable/Fiber info
    tube_color_id INT,
    core_used INT DEFAULT 0,
    core_color_id INT NULL,
    item_cable_type ENUM('backbone', 'distribution', 'drop_core', 'feeder', 'branch') NULL DEFAULT 'distribution',
    total_core_capacity INT NULL DEFAULT 24,
    
    -- Pricing info
    item_type VARCHAR(100) NULL COMMENT 'Model/Type dari item',
    item_price DECIMAL(12,2) NULL COMMENT 'Harga item dalam Rupiah',
    
    -- ODC specific fields
    odc_type ENUM('pole_mounted', 'ground_mounted') DEFAULT 'pole_mounted',
    odc_capacity INT DEFAULT 32,
    odc_ports_used INT DEFAULT 0,
    odc_main_splitter_ratio VARCHAR(10) DEFAULT '1:4',
    odc_odp_splitter_ratio VARCHAR(10) DEFAULT '1:8',
    odc_input_ports INT DEFAULT 1,
    odc_output_ports INT DEFAULT 4,
    
    -- ODP specific fields  
    odp_type ENUM('pole_mounted','wall_mounted','underground') DEFAULT 'pole_mounted',
    odp_capacity INT DEFAULT 8,
    odp_ports_used INT DEFAULT 0,
    odp_splitter_ratio VARCHAR(10) DEFAULT '1:8',
    odp_input_ports INT DEFAULT 1,
    odp_output_ports INT DEFAULT 8,
    odp_parent_odc_id INT NULL,
    
    -- ONT/HTB specific fields
    ont_serial_number VARCHAR(50) NULL,
    ont_customer_name VARCHAR(255) NULL,
    ont_customer_address TEXT NULL,
    ont_model VARCHAR(100) NULL,
    ont_connected_odp_id INT NULL,
    ont_connected_port INT NULL,
    ont_installation_type ENUM('indoor','outdoor') DEFAULT 'indoor',
    ont_service_plan VARCHAR(100) NULL,
    ont_connection_status ENUM('connected','disconnected','suspended','maintenance') DEFAULT 'connected',
    
    -- Monitoring fields
    last_ping_time TIMESTAMP NULL,
    response_time_ms INT NULL,
    
    -- SNMP fields
    snmp_enabled TINYINT(1) DEFAULT 0,
    snmp_version ENUM('1', '2c', '3') DEFAULT '2c',
    snmp_community VARCHAR(255) DEFAULT 'public',
    snmp_port INT DEFAULT 161,
    cpu_usage DECIMAL(5,2) NULL,
    memory_usage DECIMAL(5,2) NULL,
    
    -- Auto generation fields
    is_auto_generated TINYINT(1) DEFAULT 0,
    auto_generated_by_route_id INT NULL,
    
    -- General fields
    status ENUM('active', 'inactive', 'maintenance') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_item_type (item_type_id),
    INDEX idx_location (latitude, longitude),
    INDEX idx_ip_address (ip_address),
    INDEX idx_status (status),
    INDEX idx_monitoring_status (monitoring_status)
);

-- Tabel untuk menyimpan routing kabel
CREATE TABLE cable_routes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    from_item_id INT,
    to_item_id INT,
    route_coordinates TEXT,
    distance DECIMAL(8,2),
    cable_type VARCHAR(50),
    core_count INT,
    route_type ENUM('straight', 'road', 'direct') DEFAULT 'straight',
    status ENUM('planned', 'installed', 'maintenance') DEFAULT 'planned',
    auto_generate_tiang_tumpu TINYINT(1) DEFAULT 0,
    generated_tiang_tumpu_ids TEXT NULL,
    tiang_tumpu_interval_meters INT DEFAULT 30,
    total_generated_cost DECIMAL(12,2) DEFAULT 0.00,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_from_item (from_item_id),
    INDEX idx_to_item (to_item_id),
    INDEX idx_status (status)
);

-- =====================================================
-- 3. MONITORING TABLES
-- =====================================================

-- Tabel untuk menyimpan log monitoring
CREATE TABLE monitoring_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_id INT NOT NULL,
    monitoring_type ENUM('ping', 'snmp', 'both') DEFAULT 'ping',
    ping_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    status ENUM('online', 'warning', 'offline') NOT NULL,
    response_time_ms INT NULL,
    error_message TEXT NULL,
    snmp_status ENUM('success', 'failed', 'timeout', 'auth_failed') NULL,
    snmp_error TEXT NULL,
    
    INDEX idx_item_id (item_id),
    INDEX idx_ping_time (ping_time),
    INDEX idx_status (status)
);

-- Tabel untuk menyimpan interface information
CREATE TABLE device_interfaces (
    id INT PRIMARY KEY AUTO_INCREMENT,
    device_id INT NOT NULL,
    interface_index INT NOT NULL,
    interface_name VARCHAR(255) NOT NULL,
    interface_description VARCHAR(255),
    interface_type VARCHAR(100),
    mac_address VARCHAR(17),
    is_up TINYINT(1) DEFAULT 0,
    speed_mbps INT,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_device_id (device_id),
    INDEX idx_interface_name (interface_name),
    UNIQUE KEY unique_device_interface (device_id, interface_index)
);

-- =====================================================
-- 4. EXTENDED TABLES (SIMPLIFIED)
-- =====================================================

-- Server VLAN configurations
CREATE TABLE server_vlans (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_id INT NOT NULL,
    vlan_id VARCHAR(10) NOT NULL,
    vlan_name VARCHAR(100),
    ip_address VARCHAR(45),
    subnet_mask VARCHAR(45),
    description TEXT,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_item_id (item_id),
    INDEX idx_vlan_id (vlan_id)
);

-- OLT PON configurations
CREATE TABLE olt_pons (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_id INT NOT NULL,
    pon_port VARCHAR(50) NOT NULL,
    interface_id INT NULL,
    description TEXT,
    is_active TINYINT(1) DEFAULT 1,
    max_ont_count INT DEFAULT 64,
    current_ont_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_item_id (item_id),
    INDEX idx_pon_port (pon_port)
);

-- PON VLAN mappings
CREATE TABLE pon_vlans (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pon_id INT NOT NULL,
    vlan_id VARCHAR(10) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_pon_id (pon_id),
    INDEX idx_vlan_id (vlan_id)
);

-- SNMP OID mapping untuk monitoring
CREATE TABLE snmp_oid_mapping (
    id INT PRIMARY KEY AUTO_INCREMENT,
    device_type VARCHAR(50) NOT NULL,
    oid_name VARCHAR(100) NOT NULL,
    oid_value VARCHAR(255) NOT NULL,
    data_type ENUM('string', 'integer', 'gauge', 'counter', 'timeticks') NOT NULL,
    unit VARCHAR(20),
    description TEXT,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_device_type (device_type),
    INDEX idx_oid_name (oid_name)
);

-- SNMP metrics storage
CREATE TABLE snmp_metrics (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_id INT NOT NULL,
    oid_name VARCHAR(100) NOT NULL,
    metric_value TEXT,
    metric_timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_item_id (item_id),
    INDEX idx_timestamp (metric_timestamp),
    INDEX idx_oid_name (oid_name)
);

-- Default pricing settings
CREATE TABLE default_pricing (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_type VARCHAR(50) NOT NULL,
    price_value DECIMAL(12,2) NOT NULL DEFAULT 0.00,
    auto_calculate TINYINT(1) DEFAULT 1,
    description TEXT,
    is_active TINYINT(1) DEFAULT 1,
    created_by INT NULL,
    updated_by INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    UNIQUE KEY unique_setting_type (setting_type)
);

-- =====================================================
-- 5. ADD FOREIGN KEY CONSTRAINTS (SIMPLIFIED)
-- =====================================================

-- ftth_items constraints (minimal to avoid circular dependencies)
ALTER TABLE ftth_items
ADD CONSTRAINT fk_ftth_items_item_type FOREIGN KEY (item_type_id) REFERENCES item_types(id) ON DELETE SET NULL,
ADD CONSTRAINT fk_ftth_items_tube_color FOREIGN KEY (tube_color_id) REFERENCES tube_colors(id) ON DELETE SET NULL,
ADD CONSTRAINT fk_ftth_items_core_color FOREIGN KEY (core_color_id) REFERENCES tube_colors(id) ON DELETE SET NULL;

-- cable_routes constraints
ALTER TABLE cable_routes
ADD CONSTRAINT fk_cable_routes_from_item FOREIGN KEY (from_item_id) REFERENCES ftth_items(id) ON DELETE CASCADE,
ADD CONSTRAINT fk_cable_routes_to_item FOREIGN KEY (to_item_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- monitoring_logs constraints
ALTER TABLE monitoring_logs
ADD CONSTRAINT fk_monitoring_logs_item FOREIGN KEY (item_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- device_interfaces constraints
ALTER TABLE device_interfaces
ADD CONSTRAINT fk_device_interfaces_device FOREIGN KEY (device_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- server_vlans constraints
ALTER TABLE server_vlans
ADD CONSTRAINT fk_server_vlans_item FOREIGN KEY (item_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- olt_pons constraints
ALTER TABLE olt_pons
ADD CONSTRAINT fk_olt_pons_item FOREIGN KEY (item_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- pon_vlans constraints
ALTER TABLE pon_vlans
ADD CONSTRAINT fk_pon_vlans_pon FOREIGN KEY (pon_id) REFERENCES olt_pons(id) ON DELETE CASCADE;

-- snmp_metrics constraints
ALTER TABLE snmp_metrics
ADD CONSTRAINT fk_snmp_metrics_item FOREIGN KEY (item_id) REFERENCES ftth_items(id) ON DELETE CASCADE;

-- default_pricing constraints
ALTER TABLE default_pricing
ADD CONSTRAINT fk_default_pricing_created_by FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
ADD CONSTRAINT fk_default_pricing_updated_by FOREIGN KEY (updated_by) REFERENCES users(id) ON DELETE SET NULL;

-- =====================================================
-- 6. INSERT MINIMAL DEFAULT DATA
-- =====================================================

-- Insert basic item types (ONLY ESSENTIAL DATA)
INSERT INTO item_types (name, icon, color, supports_snmp) VALUES
('OLT', 'fas fa-server', '#FF6B6B', 1),
('Tiang Tumpu', 'fas fa-tower-broadcast', '#4ECDC4', 0),
('Tiang ODP', 'fas fa-project-diagram', '#45B7D1', 0),
('ODC Pole Mounted', 'fas fa-network-wired', '#96CEB4', 0),
('Tiang Joint Closure', 'fas fa-link', '#E74C3C', 0),
('ONT', 'fas fa-home', '#FFA500', 1),
('Server', 'fas fa-server', '#8E44AD', 1),
('Access Point', 'fas fa-wifi', '#3498DB', 1),
('Custom', 'fas fa-tools', '#9B59B6', 0),
('HTB', 'fas fa-home', '#FF6B9D', 1),
('ODC Cabinet', 'fas fa-box', '#F39C12', 0);

-- Insert default users (ONLY ESSENTIAL USERS)
-- Password untuk semua user: "password" (hashed dengan bcrypt)
INSERT INTO users (username, password, role, full_name, email) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'Administrator System', 'admin@ftthnms.com'),
('teknisi', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'teknisi', 'Teknisi Lapangan', 'teknisi@ftthnms.com');

-- =====================================================
-- 7. FINALIZE SETUP
-- =====================================================

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1;

-- =====================================================
-- SETUP COMPLETE
-- =====================================================
-- 
-- DEFAULT CREDENTIALS:
-- Username: admin | Password: password (Role: admin)
-- Username: teknisi | Password: password (Role: teknisi)
--
-- FEATURES INCLUDED:
-- ✅ Basic FTTH item management
-- ✅ Network topology mapping
-- ✅ Monitoring system (ping & SNMP)
-- ✅ User authentication & authorization
-- ✅ Cable routing & planning
-- ✅ ODC/ODP management
-- ✅ ONT/HTB customer management
-- ✅ Pricing & cost calculation
-- ✅ Auto-generation features
--
-- FEATURES REMOVED (for clean setup):
-- ❌ Excessive sample data
-- ❌ Complex foreign key dependencies
-- ❌ Circular reference constraints
-- ❌ Overly complex table relationships
--
-- This database can be imported without issues and 
-- provides a solid foundation for FTTH management.
-- =====================================================
